/*
|| Script:  h3-queries.sql
|| Purpose: Shows various aspects of the new 23ai H3 capabilities
|| Author:  Jim Czuprynski, Zero Defect Computing, Inc.
*/

-----
-- H3 Example #1
-- Calculate the H3_KEY and H3_BOUNDARY for specific longitude/latitude 
-- pair values at a specific resolution, retrieving only the first 10 rows
-----
WITH h3_values AS (
  SELECT 
    usgs_case_id AS case_id
  , state_abbr
  , county
  , pv_array_area
  , facility_capacity_dc
  , facility_capacity_ac
  , SDO_UTIL.H3_KEY(longitude, latitude, :res) AS h3key
    FROM existing_pv_sites)
SELECT 
  h3key
, case_id
, state_abbr
, county
, pv_array_area AS area
, facility_capacity_dc AS dc_cap
, facility_capacity_ac AS ac_cap
, SDO_UTIL.H3_BOUNDARY(h3key) AS h3geom
  FROM h3_values
 WHERE state_abbr = 'WI'
 FETCH FIRST 10 ROWS ONLY
;

-----
-- H3 Example #2
-- Summarizing values within H3_KEY derivations and then linking
-- them based on matching H3_KEY attributes (i.e. within the same
-- H3 boundary polygon) within the specified H3 resolution factor
-----
WITH pv AS (
      SELECT 
        SDO_UTIL.H3_KEY(longitude, latitude, 5) AS h3key
      , state_abbr
      , county
      , COUNT(usgs_case_id) AS cnt_pv
      , SUM(pv_array_area) AS tot_pv_area
      , SUM(facility_capacity_ac) AS tot_pv_cap_ac
      , SUM(facility_capacity_dc) AS tot_pv_cap_dc
      FROM existing_pv_sites
     GROUP BY h3key, state_abbr, county
     ORDER BY h3key)
 ,wt AS (
     SELECT 
       SDO_UTIL.H3_KEY(geometry, 5) AS h3key
     , COUNT(usgs_case_id) AS cnt_wt
     , SUM(turbine_capacity) AS tot_wt_cap
      FROM existing_wind_turbines
     GROUP BY h3key
     ORDER BY h3key)
  SELECT 
        pv.h3key 
      , pv.state_abbr
      , pv.county
      , pv.cnt_pv 
      , pv.tot_pv_area
      , pv.tot_pv_cap_ac
      , pv.tot_pv_cap_dc
      , wt.cnt_wt 
      , wt.tot_wt_cap 
   FROM pv, wt
  WHERE pv.h3key = wt.h3key
  ORDER BY pv.h3key, pv.state_abbr, pv.county;

-----
-- H3 Example #3
-- Create an H3 table from the contents of EXISTING_WIND_TURBINES,
-- including a summarization of row counts as well as minimum, 
-- maximum, and average turbine capacity calculated at each zoom level

DROP TABLE IF EXISTS h3sum_wind_turbines;
DELETE FROM user_sdo_geom_metadata WHERE table_name = 'H3SUM_WIND_TURBINES';

BEGIN
  SDO_UTIL.H3SUM_CREATE_TABLE(
    table_out    => 'H3SUM_WIND_TURBINES'
  , table_in     => 'EXISTING_WIND_TURBINES'
  , geomcol_spec => 'GEOMETRY'
  , col_spec     => '1,CNT'
  , max_h3_level => 7
  );

END;
/

SELECT * FROM h3sum_wind_turbines;


SELECT levelnum, COUNT(*) 
  FROM h3sum_wind_turbines
 GROUP BY levelnum
 ORDER BY levelnum;

select 
  levelnum
 ,sdo_util.h3_boundary(key)
 ,cnt
  from h3sum_wind_turbines;

-----
-- Test with (x,y,z) = (15,23,6) or (16,23,6) 
-----
SELECT
SDO_UTIL.H3SUM_VECTORTILE(
     H3_TABLE => 'H3SUM_WIND_TURBINES'
    ,LEVELNUM => 5 
    ,TILE_X => 15
    ,TILE_Y => 23
    ,TILE_ZOOM => 6);

-----
-- Test with (x,y,z) = (15,23,6) or (16,23,6) as valid H3 hexagon references
-----
SELECT
  -- 'application/vnd.mapbox-vector-tile' as mediatype 
   SDO_UTIL.H3SUM_VECTORTILE(
     H3_TABLE => 'H3SUM_WIND_TURBINES'
    ,LEVELNUM => 
       CASE 
         WHEN z.z <= 5              THEN z*1 
         WHEN z.z BETWEEN  6 AND  9 THEN z-1
         WHEN z.z BETWEEN 10 AND 13 THEN z-2
         WHEN z.z BETWEEN 12 AND 16 THEN z-3
         WHEN z.z BETWEEN 14 AND 19 THEN z-4 
         WHEN z.z > 20              THEN 15
       END
    ,TILE_X => :x
    ,TILE_Y => :y
    ,TILE_ZOOM => z) as VT
  FROM (SELECT :z AS z) z;

-----
-- ORDS template for H3SUM_VECTORTILE results against 
-- EXISTING_WIND_TURBINES table, including calculated
-- minimum, maximum and average turbine capacity values
-----
BEGIN
   ORDS.DELETE_MODULE('wt_summary');
END;
/

BEGIN
  ORDS.DEFINE_MODULE(
    P_MODULE_NAME => 'wt_summary',
    P_BASE_PATH => '/wt_summary/',
    P_ITEMS_PER_PAGE => 25,
    P_STATUS => 'PUBLISHED',
    P_COMMENTS => 'Wind Turbine H3 Summary Tiling REST API'
  );
  COMMIT;
END;
/

BEGIN
  ORDS.DEFINE_TEMPLATE(
    P_MODULE_NAME => 'wt_summary',
    P_PATTERN => 'vt/:z/:x/:y',
    P_PRIORITY => 0,
    P_ETAG_TYPE => 'HASH',
    P_COMMENTS => 'Wind Turbine H3 Summary Tiling Template'
  );
  COMMIT;
END;
/

BEGIN
  ORDS.DEFINE_HANDLER(
    P_MODULE_NAME => 'wt_summary'
   ,P_PATTERN => 'vt/:z/:x/:y'
   ,P_METHOD => 'GET'
   ,P_SOURCE_TYPE => ORDS.SOURCE_TYPE_MEDIA
   ,P_SOURCE => 
'SELECT
  ''application/vnd.mapbox-vector-tile'' as mediatype 
  ,SDO_UTIL.H3SUM_VECTORTILE(
     H3_TABLE => ''H3SUM_WIND_TURBINES''
    ,LEVELNUM => 
       CASE 
         WHEN z.z <= 5              THEN z*1 
         WHEN z.z BETWEEN  6 AND  9 THEN z-1
         WHEN z.z BETWEEN 10 AND 13 THEN z-2
         WHEN z.z BETWEEN 12 AND 16 THEN z-3
         WHEN z.z BETWEEN 14 AND 19 THEN z-4 
         WHEN z.z > 20              THEN 15
       END
    ,TILE_X => :x
    ,TILE_Y => :y
    ,TILE_ZOOM => z) as VT
  FROM (SELECT :z AS z) z');
END;
/

-----
BONUS: Examples of other H3 functions!
-----

-- KEY, MBR, PARENT, and RESOLUTION functions:
SELECT SDO_UTIL.H3_KEY(48.85755957774311, 2.344920508484808, 7) FROM DUAL;
SELECT SDO_UTIL.H3_MBR(SDO_UTIL.H3_KEY(48.85755957774311, 2.344920508484808, 7)) FROM DUAL;
SELECT SDO_UTIL.H3_PARENT(SDO_UTIL.H3_KEY(48.85755957774311, 2.344920508484808, 7)) FROM DUAL;
SELECT SDO_UTIL.H3_RESOLUTION(SDO_UTIL.H3_KEY(48.85755957774311, 2.344920508484808, 5)) FROM DUAL;

-- Finding base cell, boundary, and center of a particular H3 polygon:
SELECT SDO_UTIL.H3_BASE_CELL(SDO_UTIL.H3_KEY(48.85755957774311, 2.344920508484808, 5)) FROM DUAL;
SELECT SDO_UTIL.H3_BOUNDARY(SDO_UTIL.H3_KEY(48.85755957774311, 2.344920508484808, 5)) FROM DUAL;
SELECT SDO_UTIL.H3_CENTER(SDO_UTIL.H3_KEY(48.85755957774311, 2.344920508484808, 5)) FROM DUAL;

-- Calculating total number of cells within resolution (MAX resolution in this example is 15)
SELECT SDO_UTIL.H3_NUM_CELLS(15) FROM DUAL;
SELECT SDO_UTIL.H3_HEX_AREA(15, 'SQ_MILE', 'MAX') FROM DUAL;
SELECT SDO_UTIL.H3_HEX_EDGELEN(15, 'METER', 'MAX') FROM DUAL;
SELECT SDO_UTIL.H3_PENTAGON_AREA(15,'SQ_MILE') FROM DUAL;
SELECT SDO_UTIL.H3_PENTAGON_edgelen(15,'METER') FROM DUAL;

